package net.sf.saxon.functions;
import net.sf.saxon.expr.Expression;
import net.sf.saxon.expr.ExpressionVisitor;
import net.sf.saxon.expr.XPathContext;
import net.sf.saxon.om.DocumentInfo;
import net.sf.saxon.om.Item;
import net.sf.saxon.om.NodeInfo;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.ItemType;
import net.sf.saxon.type.Type;
import net.sf.saxon.value.StringValue;

/**
* Implements the unparsed-entity-uri() function defined in XSLT 1.0
* and the unparsed-entity-public-id() function defined in XSLT 2.0
*/


public class UnparsedEntity extends SystemFunction {

    public static int URI = 0;
    public static int PUBLIC_ID = 1;

    /**
    * Simplify: add a second implicit argument, the context document
     * @param visitor an expression visitor
     */

     public Expression simplify(ExpressionVisitor visitor) throws XPathException {
        UnparsedEntity f = (UnparsedEntity)super.simplify(visitor);
        f.addContextDocumentArgument(1, (operation==URI ? "unparsed-entity-uri_9999_": "unparsed-entity-public-id_9999_"));
        return f;
    }

    /**
     * Type-check the expression. This also calls preEvaluate() to evaluate the function
     * if all the arguments are constant; functions that do not require this behavior
     * can override the preEvaluate method.
     */

    public Expression typeCheck(ExpressionVisitor visitor, ItemType contextItemType) throws XPathException {
        try {
            return super.typeCheck(visitor, contextItemType);
        } catch (XPathException err) {
            if ("XPDY0002".equals(err.getErrorCodeLocalPart())) {
                if (operation == URI) {
                    XPathException e = new XPathException("Cannot call the unparsed-entity-uri()" +
                            " function when there is no context node");
                    e.setErrorCode("XTDE1370");
                    throw e;
                } else {
                    XPathException e = new XPathException("Cannot call the unparsed-entity-public-id()" +
                            " function when there is no context node");
                    e.setErrorCode("XTDE1380");
                    throw e;
                }
            }
            throw err;
        }
    }

    /**
    * preEvaluate: this method suppresses compile-time evaluation by doing nothing
     * @param visitor an expression visitor
     */

    public Expression preEvaluate(ExpressionVisitor visitor) {
        return this;
    }

    /**
    * Evaluate the expression
    */

    public Item evaluateItem(XPathContext context) throws XPathException {
        String arg0 = argument[0].evaluateItem(context).getStringValue();
        NodeInfo doc = null;
        try {
            doc = (NodeInfo)argument[1].evaluateItem(context);
        } catch (XPathException err) {
            String code = err.getErrorCodeLocalPart();
            if ("XPDY0002".equals(code)) {
                if (operation == URI) {
                    XPathException e = new XPathException("Cannot call the unparsed-entity-uri()" +
                            " function when there is no context node");
                    e.setErrorCode("XTDE1370");
                    throw e;
                } else {
                    XPathException e = new XPathException("Cannot call the unparsed-entity-public-id()" +
                            " function when there is no context node");
                    e.setErrorCode("XTDE1380");
                    throw e;
                }
            } else if ("XPDY0050".equals(code)) {
                if (operation == URI) {
                    XPathException e = new XPathException("Can only call the unparsed-entity-uri()" +
                            " function when the context node is in a tree rooted at a document node");
                    e.setErrorCode("XTDE1370");
                    throw e;
                } else {
                    XPathException e = new XPathException("Can only call the unparsed-entity-public-id()" +
                            " function when the context node is in a tree rooted at a document node");
                    e.setErrorCode("XTDE1380");
                    throw e;
                }
            }
        }
        if (doc.getNodeKind() != Type.DOCUMENT) {
            String code = (operation==URI ? "XTDE1370" : "XTDE1380");
            dynamicError("In function " + getDisplayName() +
                            ", the context node must be in a tree whose root is a document node", code, context);
        }
        String[] ids = ((DocumentInfo)doc).getUnparsedEntity(arg0);
        if (ids==null) return StringValue.EMPTY_STRING;
        return new StringValue(ids[operation]);
    }

}




//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//
