/*
 * Copyright (C) 2002-2008 The Warp Rogue Team
 * Part of the Warp Rogue Project
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY.
 *
 * See the license.txt file for more details.
 */

/*
 * Module: AI Combat Options
 */

#include "wrogue.h"


/* When the AI decides what to do during combat, it usually has to 
 * choose between multiple options.
 *
 * To do this the AI calculates the 'weight' of each option and 
 * then chooses the option with the highest weight.
 */

static AI_OPTION_WEIGHT ai_option_weight(CHARACTER *, AI_OPTION);



/* Returns the combat option chosen by the AI
 */
AI_OPTION ai_choose_option(CHARACTER *self)
{
        AI_OPTION_WEIGHT chosen_option_weight = AI_OPTION_WEIGHT_NIL;
        AI_OPTION chosen_option = AI_OPTION_NIL;
        AI_OPTION option;
        const SECTOR *sector;
        
        sector = sector_at(&self->target.point);
        if (sector->character == NULL && sector->object == NULL) {
                return AI_OPTION_NIL;
        }

        for (option = 0; option < MAX_AI_OPTIONS; ++option) {
                AI_OPTION_WEIGHT option_weight = 
                        ai_option_weight(self, option);

                if (option_weight > chosen_option_weight) {

                        chosen_option = option;
                        chosen_option_weight = option_weight;
                }
        }
        
        return chosen_option;
}



/* Executes a combat option
 */
void ai_use_option(CHARACTER * self, AI_OPTION option)
{
	if (option == AI_OPTION_UNARMED) {

                action_strike(self, NULL, &self->target.point);

        } else if (option == AI_OPTION_WEAPON ||
                option == AI_OPTION_SECONDARY_WEAPON) {
        
                if (self->weapon == NULL ||
                        self->weapon->type == OTYPE_CLOSE_COMBAT_WEAPON) {

                        action_strike(self, self->weapon, 
                                &self->target.point
                        );

                } else {
                        FIRING_DATA firing_data;

                        ai_choose_firing_mode(self, &firing_data);

                        action_shoot(self, self->weapon, &firing_data,
                                &self->target.point
                        );
                }

        } else {
                
                action_recover(self);
        }
}



/* Returns the weight of a combat option
 */
static AI_OPTION_WEIGHT ai_option_weight(CHARACTER *self, AI_OPTION option)
{
        AI_OPTION_WEIGHT option_weight = AI_OPTION_WEIGHT_NIL;

        if (option == AI_OPTION_WEAPON) {
        
                option_weight = ai_weapon_weight(self, self->weapon);

        } else if (option == AI_OPTION_SECONDARY_WEAPON) {
                
                option_weight = ai_weapon_weight(self, 
                        self->secondary_weapon
                );

        } else if (option == AI_OPTION_UNARMED) {

                option_weight = ai_weapon_weight(self, NULL);
        }

        return option_weight;
}


