/*
Copyright 2013 Aurélien Gâteau <agateau@kde.org>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) version 3, or any
later version accepted by the membership of KDE e.V. (or its
successor approved by the membership of KDE e.V.), which shall
act as a proxy defined in Section 6 of version 3 of the license.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "ktranscripttest.h"

#include <QtTest/QtTest>

#include <QDebug>

#include <ktranscript_p.h>

QTEST_MAIN(KTranscriptTest)

extern "C"
{
    typedef KTranscript *(*InitFunc)();
}

KTranscriptTest::KTranscriptTest()
    : m_transcript(0)
{
}

void KTranscriptTest::initTestCase()
{
    QString pluginPath = QStringLiteral(KTRANSCRIPT_PATH);
    QVERIFY2(QFile::exists(pluginPath), "Could not find ktranscript plugin");

    m_library.setFileName(pluginPath);

    QVERIFY(m_library.load());
    InitFunc initf = (InitFunc) m_library.resolve("load_transcript");
    QVERIFY(initf);
    m_transcript = initf();
    QVERIFY(m_transcript);
}

void KTranscriptTest::test_data()
{
    QTest::addColumn<QVariantList>("argv");
    QTest::addColumn<bool>("fallsBack");
    QTest::addColumn<QString>("expected");

    QTest::newRow("test_basic")
        << (QVariantList() << "test_basic" << "foo")
        << false
        << "foo bar";
    QTest::newRow("test_unicode")
        << (QVariantList() << "test_unicode" << "čгσィ九")
        << false
        << "čгσィ九 фу";
    QTest::newRow("test_hascall")
        << (QVariantList() << "test_hascall" << "test_basic")
        << false
        << "yes";
    QTest::newRow("test_acall")
        << (QVariantList() << "test_acall" << "test_basic" << "qwyx")
        << false
        << "qwyx bar";
    QTest::newRow("test_load")
        << (QVariantList() << "test_load")
        << false
        << "foo blurb";
    QTest::newRow("test_fallback")
        << (QVariantList() << "test_fallback")
        << true
        << "";
    QTest::newRow("test_msgid")
        << (QVariantList() << "test_msgid")
        << false
        << "source-text";
    QTest::newRow("test_msgtrf")
        << (QVariantList() << "test_msgtrf")
        << false
        << "translated-text";
    QTest::newRow("test_msgctxt")
        << (QVariantList() << "test_msgctxt")
        << false
        << "a-context";
    QTest::newRow("test_msgkey")
        << (QVariantList() << "test_msgkey")
        << false
        << "a-context|source-text";
    QTest::newRow("test_nsubs")
        << (QVariantList() << "test_nsubs")
        << false
        << "2";
    QTest::newRow("test_subs")
        << (QVariantList() << "test_subs" << 1)
        << false
        << "qwyx";
    QTest::newRow("test_vals")
        << (QVariantList() << "test_vals" << 0 << 5)
        << false
        << "50";
    QTest::newRow("test_dynctxt")
        << (QVariantList() << "test_dynctxt" << "origin")
        << false
        << "neverwhere";
    QTest::newRow("test_dbgputs")
        << (QVariantList() << "test_dbgputs")
        << false
        << "debugged";
    QTest::newRow("test_warnputs")
        << (QVariantList() << "test_warnputs")
        << false
        << "warned";
    QTest::newRow("test_setcallForall")
        << (QVariantList() << "test_setcallForall")
        << false
        << "done";
    QTest::newRow("test_toUpperFirst")
        << (QVariantList() << "test_toUpperFirst" << "...123 foo")
        << false
        << "...123 Foo";
    QTest::newRow("test_toUpperFirst_unicode")
        << (QVariantList() << "test_toUpperFirst" << "...123 фу")
        << false
        << "...123 Фу";
    QTest::newRow("test_toLowerFirst")
        << (QVariantList() << "test_toLowerFirst" << "...123 FOO")
        << false
        << "...123 fOO";
    QTest::newRow("test_loadProps")
        << (QVariantList() << "test_loadProps" << "cities")
        << false
        << "loaded";
    QTest::newRow("test_getProp")
        << (QVariantList() << "test_getProp" << "cities" << "Athens" << "gen")
        << false
        << "Atine";
    QTest::newRow("test_setProp")
        << (QVariantList() << "test_setProp" << "Oslo" << "dat" << "Oslou")
        << false
        << "Oslou";
    QTest::newRow("test_normKey")
        << (QVariantList() << "test_normKey" << "Some &Thing")
        << false
        << "something";
    // TODO: How to test getConf* functions?
    // Enable setting user configuration path from environment variable?
}

void KTranscriptTest::test()
{
    QFETCH(QVariantList, argv);
    QFETCH(bool, fallsBack);
    QFETCH(QString, expected);

    QString language = "fr";
    QString country = "fr";
    QString msgctxt = "a-context";
    QHash<QString, QString> dynamicContext;
    dynamicContext.insert("origin", "neverwhere");
    QString msgid = "source-text";
    QStringList subs;
    subs << "10" << "qwyx";
    QList<QVariant> values;
    values << 10 << "qwyx";
    QString ordinaryTranslation = "translated-text";

    QString testJs = QFINDTESTDATA("test.js");
    QList<QStringList> modules;
    modules << (QStringList() << testJs << language);

    QString error;
    bool fallback;
    QString result = m_transcript->eval(
                         argv, language, country,
                         msgctxt, dynamicContext, msgid,
                         subs, values, ordinaryTranslation,
                         modules, error, fallback);

    if (!error.isEmpty()) {
        QFAIL(qPrintable(error));
    }
    if (!fallsBack) {
        QVERIFY(!fallback);
        QCOMPARE(result, expected);
    } else {
        QVERIFY(fallback);
    }
}

