/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.util.Map;

import org.apache.log4j.Level;
import org.jrobin.core.RrdDb;
import org.jrobin.core.RrdDbPool;
import org.jrobin.core.RrdException;
import org.jrobin.core.Sample;

import snmp.SNMPBadValueException;
import snmp.SNMPGetException;
import snmp.SNMPSequence;
import snmp.SNMPVarBindList;
import snmp.SNMPv1CommunicationInterface;

/**
 * Implementation of IVlServer.
 * 
 * @author SylV
 * @version 1.0
 * 
 */
public class VlServerReal extends VlServer {

    public VlServerReal(String name, InetAddress ip) {
        super(name, ip);
    }

    /**
     * Updates the RRD data according to the SNMP responses.
     */
    public void updateSnmpData() {
        String rrdFile = createRrdFileIfNecessary();

        try {
            SNMPv1CommunicationInterface comm = new SNMPv1CommunicationInterface(
                    0, getIp(), Configuration.getInstance().getSnmpCommunity());
            Map<String, String> configurationOids  = Configuration.getInstance().getoids();
            String[] oid = { configurationOids.get("CPU_LOAD"), 
                    configurationOids.get("TRAFFIC_IN"),
                    configurationOids.get("TRAFFIC_OUT"), 
                    configurationOids.get("VLC_CPU"), 
                    configurationOids.get("VLC_MEM") };
            SNMPVarBindList varList = comm.getMIBEntry(oid);
            SNMPSequence pair = (SNMPSequence) (varList.getSNMPObjectAt(0));
            currentCpuLoad = Double.parseDouble(pair.getSNMPObjectAt(1)
                    .toString().trim());
            pair = (SNMPSequence) (varList.getSNMPObjectAt(1));
            currentTrafficIn = Long.parseLong(pair.getSNMPObjectAt(1)
                    .toString().trim());
            pair = (SNMPSequence) (varList.getSNMPObjectAt(2));
            currentTrafficOut = Long.parseLong(pair.getSNMPObjectAt(1)
                    .toString().trim());
            pair = (SNMPSequence) (varList.getSNMPObjectAt(3));
            currentVlcCpu = Double.parseDouble(pair.getSNMPObjectAt(1)
                    .toString().trim());
            pair = (SNMPSequence) (varList.getSNMPObjectAt(4));
            currentVlcMem = Double.parseDouble(pair.getSNMPObjectAt(1)
                    .toString().trim());
        } catch (SNMPGetException e) {
            logger.error(
                    "Error while fetching data from SNMP of "
                            + getName(), e);
        } catch (SNMPBadValueException e) {
            logger.error(
                    "Error while fetching data from SNMP of "
                            + getName(), e);
        } catch (IOException e) {
            logger.error(
                    "Error while fetching data from SNMP of "
                            + getName(), e);
        } catch (NumberFormatException e) {
            logger.error(
                    "Error while fetching data from SNMP of "
                            + getName(), e);
        } catch (Exception e) {
            logger.error(
                    "Error while fetching data from SNMP of "
                            + getName(), e);
        }

        try {
            RrdDbPool rrdPool = RrdDbPool.getInstance();
            RrdDb rrdDb = rrdPool.requestRrdDb(rrdFile);
            Sample sample = rrdDb.createSample();
            // sample.setAndUpdate(Util.getTime() + ":" + cpuLoadString)
            sample.setValue("cpu_load", currentCpuLoad);
            sample.setValue("traffic_in", currentTrafficIn);
            sample.setValue("traffic_out", currentTrafficOut);
            sample.setValue("vlc_cpu", currentVlcCpu);
            sample.setValue("vlc_mem", currentVlcMem);
            sample.update();
            rrdPool.release(rrdDb);
        } catch (RrdException e) {
            logger.error(
                    "Error while writing data from SNMP of " + getName(), e);

        } catch (IOException e) {
            logger.error(
                    "Error while writing data from SNMP of " + getName(), e);

        }
    }

    /**
     * Verifies the accessibility of the server.
     * Updates the server state according to the result of the test.
     * 
     * @return the server accessibility: true if accessible, false otherwise
     */
    public boolean checkVLC() {
        try {
            Socket socket = new Socket(ip, Configuration.getInstance().getVlcTelnetPort());
            socket.close();
            setUp(true);
            logger.log(Level.DEBUG, "Le VLC de " + getName()
                    + " est accessible");
        } catch (IOException e) {
            logger.log(Level.ERROR, "Impossible to contact VLC server of "
                    + getName() + " through telnet interface.", e);
            setUp(false);
        }
        return isUp;
    }

    /**
     * Compares two VLC servers. 
     * 
     * @param o the object to compare
     * @return true if objects are equal
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlServerReal.class)) {
            return ((VlServerReal) o).name.equals(this.name);
        } else {
            return false;
        }
    }

    /**
     * Returns the object hashcode.
     * 
     * @return the object hashcode
     */
    public int hashCode() {
        return name.hashCode();
    }
}
